#include "modbusmaster.h"
#include <QVariant>
#include <QThread>
#include <QDebug>
#include <QtSerialPort/QSerialPort>

ModbusMaster::ModbusMaster(QObject *parent)
    : QObject(parent)
    , m_timer(new QTimer(this))
    , m_modbus(new QModbusRtuSerialMaster(this))
    , m_output0(false)
    , m_output1(false)
    , m_output2(false)
    , m_output3(false)
{
    m_modbus->setConnectionParameter(QModbusDevice::SerialPortNameParameter, "ttyS0");
    m_modbus->setConnectionParameter(QModbusDevice::SerialBaudRateParameter, QSerialPort::Baud115200);
    m_modbus->setConnectionParameter(QModbusDevice::SerialParityParameter, QSerialPort::NoParity);
    m_modbus->setConnectionParameter(QModbusDevice::SerialDataBitsParameter, QSerialPort::Data8);
    m_modbus->setConnectionParameter(QModbusDevice::SerialStopBitsParameter, QSerialPort::OneStop);
    m_modbus->setTimeout(50);
    m_modbus->setNumberOfRetries(1);

    if (!m_modbus->connectDevice())
    {
        qDebug() << "connectDevice: " << m_modbus->errorString();
    }

    m_timer->setInterval(50);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(run()));
    m_timer->start();
}

ModbusMaster::~ModbusMaster()
{
    m_modbus->disconnectDevice();

    delete m_modbus;
    delete m_timer;
}

void ModbusMaster::run()
{
    // Write Outputs
    QModbusDataUnit write(QModbusDataUnit::Coils, 32, 4);
    write.setValue(0, m_output0);
    write.setValue(1, m_output1);
    write.setValue(2, m_output2);
    write.setValue(3, m_output3);
    if (auto *reply = m_modbus->sendWriteRequest(write, 1))
    {
        if (!reply->isFinished())
        {
            connect(reply, SIGNAL(finished()), this, SLOT(processWrite()));
        }
        else
        {
            delete reply;
        }
    }

    // Read Inputs
    QModbusDataUnit read(QModbusDataUnit::DiscreteInputs, 8, 4);
    if (auto* reply = m_modbus->sendReadRequest(read, 1))
    {
        if (!reply->isFinished())
        {
            connect(reply, SIGNAL(finished()), this, SLOT(processRead()));
        }
        else
        {
            delete reply;
        }
    }
}

void ModbusMaster::processRead()
{
    auto reply = qobject_cast<QModbusReply*>(sender());
    if (reply->error() != QModbusDevice::NoError)
    {
        qDebug() << "sendReadRequest: " << reply->errorString();
    }
    else
    {
        auto result = reply->result();
        emit input0(result.value(3));
        emit input1(result.value(2));
        emit input2(result.value(1));
        emit input3(result.value(0));
    }

    delete reply;
}

void ModbusMaster::processWrite()
{
    auto reply = qobject_cast<QModbusReply*>(sender());
    if (reply->error() != QModbusDevice::NoError)
    {
        qDebug() << "sendWriteRequest: " << reply->errorString();
    }

    delete reply;
}

void ModbusMaster::output0(bool value)
{
    m_output0 = value;
}

void ModbusMaster::output1(bool value)
{
    m_output1 = value;
}

void ModbusMaster::output2(bool value)
{
    m_output2 = value;
}

void ModbusMaster::output3(bool value)
{
    m_output3 = value;
}
